#ifndef JOBQUEUE_H
#define JOBQUEUE_H

#include <Core/Core.h>

namespace Upp {

typedef Function<bool(void)> Job;

class JobQueue : Moveable<JobQueue> {
public:
    virtual bool                Do();
    virtual bool                Execute();

    bool                        InProgress() const                { return state == INPROGRESS || state == CLEANUP;  }
    bool                        IsSuccess() const                 { return state == SUCCESS; }
    bool                        IsFailure() const                 { return state == FAILURE;  }

    void                        GlobalTimeout(int ms)             { timeout = Nvl(ms, 0); }
    Event<>                     WhenDo;
    Event<>                     WhenHalt;

    int                         GetError() const                  { return error_code; }
    const String&               GetErrorDesc() const              { return error_desc; }

    JobQueue();
    virtual ~JobQueue() {};
    JobQueue(JobQueue&& jq) = default;

protected:
    Job&                        AddJob()                         { return queue.Add(); }
    JobQueue&                   AddJob(const Job& jb)            { queue.Add(jb); return *this; }
    Job&                        InsertJob(int i)                 { return queue.Insert(i); }
    JobQueue&                   InsertJob(int i,const Job& jb)   { queue.Insert(i, jb); return *this; }
    Job&                        GetJob(int i = 0)                { return queue.At(i); }
    void                        RemoveJob(int i)                 { queue.Remove(i); }
    void                        NextJob()                        { if(!queue.IsEmpty()) queue.Remove(0); }
    void                        ClearQueue()                     { if(!queue.IsEmpty()) queue.Clear(); state = FAILURE; start_time = 0; error_code = 0; }
    bool                        QueueHasJobs() const             { return queue.GetCount() > 1; }
    bool                        QueueIsHalted() const            { return state == FAILURE; }
    bool                        QueueIsEmpty() const             { return queue.IsEmpty(); }
    int                         GetJobCount() const              { return queue.GetCount(); }
    inline void                 Halt(const char* reason = NULL)  { Halt(-1, reason); }
    void                        Halt(int rc, const char* reason = NULL);
    bool                        IsTimeout() const;
    bool                        IsCleanup() const                { return state == CLEANUP; }

private:
    void                        CleanUp();
    Vector<Job> queue;
    int         start_time;
    int         timeout;
    int         error_code;
    String      error_desc;
    int         state;
    enum { FAILURE, SUCCESS, INPROGRESS, CLEANUP };
};
}

#endif
